"""This module provides classes and functions for handling I/O operations in Python."""

from __future__ import annotations

from collections import Counter
from copy import deepcopy
from enum import EnumMeta as EnumType
from inspect import Parameter
from typing import Any, TYPE_CHECKING

from altair_aitools.ext._utils._types import unpack_annotation

from ._crypto import decrypt_connection_params
from ._types import FileType
from .conversion import file_converter
from .data_exchange import DataExchangeHandler, FileSystemDataExchangeHandler

if TYPE_CHECKING:
    from ..wrapper.wrapper_json_parser import Functions, Input, Output

SUPPORTED_DATA_EXCHANGE_MODES: dict[str, DataExchangeHandler.__class__] = {
    "file_system": FileSystemDataExchangeHandler
}


class PythonIO:
    """Provides a mechanism for Python operator I/O."""

    def __init__(self, counter: Counter):
        """Initializes a PythonIO instance.

        Args:
            counter (Counter): A Counter object to track input usage.
        """
        self._counter = counter
        self._inputs = {}

    def __contains__(self, input_key: str) -> bool:
        return input_key in self._inputs

    def __getitem__(self, input_key: str) -> Any:
        input_item = self._inputs[input_key]
        self._counter[input_key] -= 1
        if self._counter[input_key]:
            input_item = deepcopy(input_item)
        else:
            del self._inputs[input_key]
        return input_item

    def __setitem__(self, input_key: str, input_item: Any) -> None:
        """Sets an input value.

        Args:
            input_key (str): The input key.
            input_item (Any): The input value.
        """
        self._inputs[input_key] = input_item


class IOHandler:
    """Wraps the business logic of I/O parsing."""

    def __init__(self, functions: Functions):
        """Initializes an IOHandler.

        Args:
            functions (Functions): Functions to be executed.
        """
        data_exchange = functions.data_exchange
        if data_exchange.mode not in SUPPORTED_DATA_EXCHANGE_MODES:
            raise ValueError(f"Unsupported data exchange mode '{data_exchange.mode}'")
        data_exchange_initializer = SUPPORTED_DATA_EXCHANGE_MODES[data_exchange.mode]
        self.data_exchange_handler = data_exchange_initializer(data_exchange.parameters)
        python_input_counter = Counter(
            input.value
            for function in functions
            for input in function.inputs
            if input.source == "python"
        )
        self.python_inputs = PythonIO(python_input_counter)

    def get_input(self, input: Input, parameter_sig: Parameter) -> Any:
        """Parses an operator input.

        Args:
            input (Input): Input value and its properties.
            parameter_sig (Parameter): Parameter signature of the input.

        Returns:
            Any: Parsed input value.
        """
        param_type, inner_types, _, _ = unpack_annotation(parameter_sig.annotation)
        if param_type is list:
            param_type = inner_types[0]
        if input.source == "default":
            if parameter_sig.default == Parameter.empty:
                raise ValueError(f"Input parameter '{input.name}' has no default value")
            return parameter_sig.default
        elif input.source == "provided":
            if input.data_type == "connection":
                if not isinstance(input.value, list):
                    raise ValueError(
                        "The input value '{input.name}' cannot be parsed as a Connection"
                    )
                from ..wrapper.wrapper_json_parser import ConnectionParam
                return param_type(
                    decrypt_connection_params(
                        [ConnectionParam.model_validate(v) for v in input.value]
                    )
                )
            return handle_enum_conversion(input.value, parameter_sig.annotation)
        elif input.source == "python":
            if (
                    not isinstance(input.value, str)
                    or input.value not in self.python_inputs
            ):
                raise ValueError(
                    f"The input value '{input.value}' was not output by a previous Python operator."
                )
            value = self.python_inputs[input.value]
            if issubclass(param_type, FileType):
                value = file_converter(value, param_type, input.collection)
            return value
        elif input.source == "rapidminer":
            return self.data_exchange_handler.load_input(input, param_type)
        else:
            raise ValueError(f"Unsupported input source: '{input.source}'")

    def process_output(self, output_meta: Output, output: Any) -> None:
        """Processes an operator output.

        Args:
            output_meta (Output): Properties of the output.
            output (Any): Output value.
        """
        if output_meta.target == "discard":
            pass
        elif output_meta.target == "python":
            self.python_inputs[output_meta.name] = output
        elif output_meta.target == "rapidminer":
            self.data_exchange_handler.return_output(output_meta, output)
        else:
            raise ValueError(f"Unsupported output target: '{output_meta.target}'")


def handle_enum_conversion(value: Any, input_type: type) -> Any:
    """Handles conversion of values to their corresponding enum or collection types.

    Args:
        value (Any): The value to convert.
        input_type (type): The type to which the value should be converted.

    Returns:
        Any: The converted value.

    Raises:
        ValueError: If the value cannot be converted to the specified type.
    """
    param_type, inner_types, _, _ = unpack_annotation(input_type)
    if isinstance(param_type, EnumType):
        if not isinstance(value, str) or value not in param_type.__members__:
            raise ValueError(
                f"Value '{value}' cannot be converted to enum '{param_type.__name__}'"
            )
        return param_type[value]
    elif param_type is list:
        if not isinstance(value, list):
            raise ValueError(
                f"Value '{value}' cannot be converted to list of '{inner_types[0]}'"
            )
        return [handle_enum_conversion(v, inner_types[0]) for v in value]
    elif param_type is tuple:
        if not isinstance(value, list) or len(value) != len(inner_types):
            raise ValueError(
                f"Value '{value}' cannot be converted to tuple of '{inner_types}'"
            )
        return tuple(handle_enum_conversion(v, t) for v, t in zip(value, inner_types))
    elif param_type is dict:
        if not isinstance(value, dict):
            raise ValueError(f"Value '{value}' cannot be converted to dict")
        return {
            handle_enum_conversion(k, inner_types[0]): handle_enum_conversion(
                v, inner_types[1]
            )
            for k, v in value.items()
        }
    return value
