"""Provides a in-memory caching mechanism for Python operator I/O."""

import sys
from copy import deepcopy
from typing import Any


class IOCache:
    """Provides a n in-memory aching mechanism for Python operator I/O."""

    def __init__(self):
        """Initializes a IOCache instance."""
        self._inputs: dict[str, Any] = {}
        self._ttl_counter: dict[str, int] = {}

    def __contains__(self, input_key: str) -> bool:
        return input_key in self._inputs

    def __getitem__(self, input_key: str) -> Any:
        input_item = self._inputs[input_key]
        self._ttl_counter[input_key] -= 1
        if self._ttl_counter[input_key]:
            input_item = deepcopy(input_item)
        else:
            del self[input_key]
        return input_item

    def __setitem__(self, input_key: str, input_item: Any) -> None:
        self._inputs[input_key] = input_item
        # TTL defaults to the maximum integer value
        self._ttl_counter[input_key] = sys.maxsize

    def __delitem__(self, input_key: str) -> None:
        del self._inputs[input_key]
        del self._ttl_counter[input_key]

    def set_ttl(self, input_key: str, max_hits: int) -> None:
        """Sets the time-to-live for a cached item.

        Args:
            input_key: The key of the cached item.
            max_hits: The maximum number of hits before the item is removed from the cache.
        """
        self._ttl_counter[input_key] = max_hits

    def clear(self) -> None:
        """Clears the cache."""
        self._inputs.clear()
        self._ttl_counter.clear()
