"""Module for converting between different types."""

from io import BytesIO, TextIOWrapper
from pathlib import Path
from typing import Any

from ._types import FileType


def file_converter(
    object: Any, target_type: type, collection: bool
) -> FileType | list[FileType]:
    """Converts a file object to a different file type.

    Args:
        object (Any): The object to convert.
        target_type (type): The target file type.
        collection (bool): Whether the object is a collection.

    Returns:
        FileType | list[FileType]: The converted object.
    """
    if collection:
        if not isinstance(object, list):
            raise TypeError(f"Expected a list, got {type(object)}")
        return [_convert_file(item, target_type) for item in object]
    else:
        return _convert_file(object, target_type)


def _convert_file(object: Any, target_type: type) -> FileType:
    if not isinstance(object, FileType):
        raise TypeError(f"Expected a file type, got {type(object)}")
    if isinstance(object, target_type):
        return object
    if isinstance(object, Path):
        # Convert Path to BytesIO and return early if the target type is BytesIO
        with open(object, "rb") as file:
            object = BytesIO(file.read())
        if target_type == BytesIO:
            return object
    if isinstance(object, BytesIO) and target_type == TextIOWrapper:
        # Convert BytesIO to TextIOWrapper
        return TextIOWrapper(object, encoding="utf-8")
    if isinstance(object, TextIOWrapper) and target_type == BytesIO:
        # Convert TextIOWrapper to BytesIO
        return BytesIO(object.read().encode("utf-8"))
    raise TypeError(f"Cannot convert {type(object)} to {target_type}")
