"""Utility functions for the PEW package."""

from __future__ import annotations

import logging
import sys
from pathlib import Path
from urllib.parse import urlparse
from urllib.request import url2pathname


def setup_logging(verbose: bool, *handlers: logging.Handler) -> None:
    """Configures logging to the standard output.

    Args:
        verbose (bool): If True, sets the logging level to DEBUG; otherwise, sets it to INFO.
        handlers (logging.Handler): Additional handlers to be added to the root logger.
    """
    root = logging.getLogger()
    root.setLevel(logging.DEBUG if verbose else logging.INFO)
    default_handler = logging.StreamHandler(sys.stdout)
    root.addHandler(default_handler)
    for handler in handlers:
        root.addHandler(handler)


def get_input_file(folder: Path, file: Path | str) -> Path:
    """Returns the path of a file specified by its folder and filename.

    Args:
        folder (Path): Path of the input folder.
        file (Union[Path, str]): Filename.

    Raises:
        FileNotFoundError: If the expected file does not exist.

    Returns:
        Path: Path of the file.
    """
    location = folder / file
    if not location.exists():
        raise FileNotFoundError(f"Input file '{file}' not found in {folder.absolute()}")
    return location


def check_directory(path: Path) -> None:
    """Checks whether a directory exists and creates it otherwise.

    Args:
        path (Path): The path to the directory to be checked/created.

    Raises:
        NotADirectoryError: If the provided path points to a file instead of a directory.
    """
    if not path.exists():
        logging.debug(
            f"Directory at {path.absolute()} does not exists, so it will be created"
        )
        path.mkdir(parents=True, exist_ok=True)
    elif not path.is_dir():
        raise NotADirectoryError(
            f"The path '{path.absolute()}' points to a file, not a directory."
        )


def uri_to_path(uri: str) -> Path:
    """Converts a URI to a Path object.

    Args:
        uri (str): The URI to be converted.
    """
    return Path(url2pathname(urlparse(uri).path))
