"""Data Transfer Objects for WebSocket communication."""

from __future__ import annotations

import os
from enum import Enum
from typing import TYPE_CHECKING, Any, Optional

if TYPE_CHECKING:
    import logging
    from types import TracebackType

from pydantic import BaseModel


class Client2ServerType(Enum):
    """Message type for Client-to-Server requests."""

    START = "START"
    STOP = "STOP"


class Server2ClientType(Enum):
    """Message type for Server-to-Client requests."""

    STARTED = "STARTED"
    STOPPED = "STOPPED"
    OPERATOR_COMPLETED = "OPERATOR_COMPLETED"
    FINISHED = "FINISHED"
    LOG = "LOG"
    ERROR = "ERROR"


class TerminationParams(BaseModel):
    """Execution termination parameters."""

    execution_id: str
    patience: float


class Record(BaseModel):
    """Represents message structure of server-side events."""

    module: str
    function: str
    pathname: str
    filename: str
    lineno: int


class LogRecord(Record):
    """Represents message structure of logs."""

    level: str
    message: str

    def __init__(self, log_record: logging.LogRecord):
        """Initializes a LogRecord instance.

        Args:
            log_record (logging.LogRecord): Log record to be sent.
        """
        super().__init__(
            module=log_record.module,
            function=log_record.funcName,
            pathname=log_record.pathname,
            filename=log_record.filename,
            lineno=log_record.lineno,
            level=log_record.levelname,
            message=log_record.getMessage(),
        )


class ErrorStackRecord(Record):
    """Represents message structure of error stack records."""

    def __init__(self, tb: TracebackType):
        """Initializes an ErrorStackRecord instance.

        Args:
            tb (TracebackType): Traceback object.
        """
        # Docs: https://docs.python.org/3/reference/datamodel.html#frame-objects
        super().__init__(
            module=tb.tb_frame.f_globals.get("__name__", ""),
            function=tb.tb_frame.f_code.co_name,
            pathname=tb.tb_frame.f_code.co_filename,
            filename=os.path.basename(tb.tb_frame.f_code.co_filename),
            lineno=tb.tb_lineno,
        )


class ExceptionWrapper(BaseModel):
    """Represents message structure of exceptions."""

    classname: str
    stack_trace: list[ErrorStackRecord]

    def __init__(self, e: Exception, skip_frames: int = 0):
        """Initializes an ExceptionWrapper instance.

        Args:
            e (Exception): Exception to be wrapped.
            skip_frames (int, optional): Number of frames to skip in the stack trace. Defaults to 0.
        """
        super().__init__(classname=e.__class__.__name__, stack_trace=[])
        if e.__traceback__:
            self.extract_stack_trace(e.__traceback__, skip_frames)

    def extract_stack_trace(self, tb: TracebackType, skip_frames: int) -> None:
        """Extracts stack trace from the traceback object.

        Args:
            tb (TracebackType): Traceback object.
            skip_frames (int): Number of frames to skip in the stack trace.
        """
        while tb.tb_next:
            tb = tb.tb_next
            if skip_frames > 0:
                skip_frames -= 1
                continue
            self.stack_trace.append(ErrorStackRecord(tb))


class WebSocketRequest(BaseModel):
    """Represents message structure of Client-to-Server requests."""

    type: Client2ServerType
    data: dict[str, Any]


class WebSocketResponse(BaseModel):
    """Represents message structure of Server-to-Client requests."""

    type: Server2ClientType
    execution_id: Optional[str]
    message: Optional[str] = None
    data: Optional[dict[str, Any]] = None
