"""Wrapper class for Python operators."""

import inspect
import logging
import random
import sys
from importlib import import_module
from typing import Any, Dict, Iterator, List, Mapping, Tuple

from ..io import IOHandler
from .wrapper_json_parser import Functions, Input, Output

logger = logging.getLogger("PEW")


class FunctionWrapper:
    """A Wrapper class that takes care of potential input and output conversion(s) and executes the function(s) for the Python operator(s) with the provided parameters/inputs."""

    def executor(self, functions: Functions) -> Iterator[str]:
        """Makes function calls with the required input arguments and collects desired output in the specified folder.

        Enables cooperative multitasking by yielding control to the caller between operators.

        Args:
            functions (Functions): The functions to be executed along with their parameters/inputs.

        Yields:
            str: The name of the function that was executed.
        """
        self.io_handler = IOHandler(functions)
        random.seed(int(functions.environment["initial_random_seed"]))

        # Add the extension zip files to the system path
        logger.debug("Adding provided lib_zips to path")
        for lib in functions.lib_zips:
            logger.debug("Adding lib %s", lib)
            sys.path.insert(0, lib)
        logger.debug("All lib_zips added to path")

        for each_function in functions:
            logger.debug(f"Function inputs: {each_function.inputs}")
            func = getattr(import_module(each_function.module), each_function.function)
            parameters = inspect.signature(func).parameters
            function_inputs = self._get_inputs(each_function.inputs, parameters)
            logger.debug(
                f"Called function '{each_function.function}' with {len(function_inputs)} arguments"
            )

            if any(
                p.kind == inspect.Parameter.VAR_KEYWORD for p in parameters.values()
            ):
                generated_random_seed = random.randint(
                    int(-6e23), int(6e23)
                )  # Easter egg: Avogadro's number
                function_outputs = func(
                    **function_inputs,
                    random_seed=generated_random_seed,
                    **functions.additional_func_args,
                )
            else:
                function_outputs = func(**function_inputs)

            # Ensure output is a tuple
            if not isinstance(function_outputs, tuple):
                function_outputs = (function_outputs,)

            outputs = zip(each_function.outputs, function_outputs)
            logger.debug(f"Function outputs: {outputs}")
            self._process_outputs(outputs)
            yield each_function.function

    def execute_functions(self, functions: Functions) -> None:
        """Makes function calls with the required input arguments and collects desired output in the specified folder.

        Args:
            functions (Functions): The functions to be executed along with their parameters/inputs.
        """
        for _ in self.executor(functions):
            pass

    def _get_inputs(
        self, inputs: List[Input], parameters: Mapping[str, inspect.Parameter]
    ) -> Dict[str, Any]:
        """Gets inputs for all input types.

        Args:
            inputs (List[Input]): List of input details.
            parameters: (Mapping[str, inspect.Parameter]): Function parameters.

        Returns:
            Dict[str, Any]: Dictionary containing the inputs for the function.
        """
        return {
            input.name: self.io_handler.get_input(input, parameters[input.name])
            for input in inputs
        }

    def _process_outputs(self, function_outputs: Iterator[Tuple[Output, Any]]) -> None:
        """Processes outputs of a function based on their corresponding action.

        Args:
            function_outputs (Iterator[Tuple[Output, Any]]):
                Output definitions and actual outputs zipped together.
        """
        for output_meta, output in function_outputs:
            self.io_handler.process_output(output_meta, output)
