"""Module for parsing JSON files with functions data."""

from typing import Dict, Iterator, List, Union

from pydantic import BaseModel


class Variable(BaseModel):
    """Class for storing function variable."""

    name: str
    data_type: str
    collection: bool


class Input(Variable):
    """Class for storing function input."""

    source: str
    value: Union[int, float, str, bool, Dict, List, None]

    def __str__(self):
        return (
            f"{self.name} (type: {self.data_type}, source: {self.source}): {self.value}"
        )


class Output(Variable):
    """Class for storing function output."""

    target: str

    def __str__(self):
        return f"{self.name} ({self.data_type}): {self.target}"

class ConnectionParam(BaseModel):
    """Class for storing connection parameter."""

    key: str
    value: str
    encrypted: bool

    def __str__(self):
        return f"{self.key}{' (encrypted)' if self.encrypted else ''}: {self.value}"


class FunctionCall(BaseModel):
    """Class for storing data for function call."""

    function: str
    module: str
    inputs: List[Input]
    outputs: List[Output]

    def __str__(self):
        return f"{self.function}.{self.module}, inputs: {str(self.inputs)}, outputs: {str(self.outputs)}"


class DataExchange(BaseModel):
    """Class for storing data exchange."""

    mode: str
    parameters: Dict[str, str]

    def __str__(self):
        return f"mode: {self.mode}, parameters: {self.parameters}"


class Functions(BaseModel):
    """Class for storing all functions."""

    functions: List[FunctionCall]
    data_exchange: DataExchange
    environment: Dict[str, Union[str, int, float, bool, None]]
    additional_func_args: Dict[str, Union[str, int, float, bool, None]]
    lib_zips: List[str]

    def __iter__(self) -> Iterator[FunctionCall]:
        for f in self.functions:
            yield f

    def __str__(self):
        return f"{self.functions}, data_exchange: {self.data_exchange}, environment: {self.environment}, additional args: {self.additional_func_args}, libs: {self.lib_zips}"
